﻿#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Diagnose / (Re)Seed der Weapon-Tabelle in game.db.
Verbesserte Ausgabe: zeigt Schema + erste Zeilen + Exceptions.
"""
import argparse
import os
import shutil
import sqlite3
import sys
import traceback

DB_FILE = "game.db"

def abs_db_path():
    return os.path.abspath(DB_FILE)

def list_tables(conn):
    cur = conn.cursor()
    cur.execute("SELECT name, type FROM sqlite_master WHERE type IN ('table','view') ORDER BY name")
    return cur.fetchall()

def table_count(conn, table):
    cur = conn.cursor()
    try:
        cur.execute(f"SELECT COUNT(*) FROM {table}")
        return cur.fetchone()[0]
    except Exception:
        return None

def pragma_table_info(conn, table):
    cur = conn.cursor()
    try:
        cur.execute(f"PRAGMA table_info({table})")
        return cur.fetchall()
    except Exception:
        return None

def show_table_sample(conn, table, limit=10):
    cur = conn.cursor()
    try:
        cur.execute(f"SELECT * FROM {table} LIMIT {limit}")
        cols = [d[0] for d in cur.description] if cur.description else []
        rows = cur.fetchall()
        return cols, rows
    except Exception:
        return None, None

def backup_db(path):
    bak = path + ".bak"
    shutil.copy2(path, bak)
    print(f"Backup erstellt: {bak}")

def reset_weapon_table(path):
    print("Weapon-Tabelle wird gelöscht und neu angelegt (Seed wird ausgeführt).")
    conn = sqlite3.connect(path)
    cur = conn.cursor()
    cur.execute("DROP TABLE IF EXISTS Weapon")
    conn.commit()
    conn.close()
    import importlib
    if "db" in sys.modules:
        importlib.reload(sys.modules["db"])
        db = sys.modules["db"]
    else:
        db = importlib.import_module("db")
    db.init_db()
    print("Neu seeden abgeschlossen.")

def main():
    p = argparse.ArgumentParser()
    p.add_argument("--show", action="store_true", help="Nur anzeigen (default).")
    p.add_argument("--reset", action="store_true", help="Weapon-Tabelle löschen und neu seeden (erst Backup).")
    args = p.parse_args()

    cwd = os.getcwd()
    path = abs_db_path()
    print(f"Arbeitsverzeichnis: {cwd}")
    print(f"DB-Pfad (relativ): {DB_FILE}")
    print(f"DB-Pfad (absolut): {path}")
    exists = os.path.exists(path)
    print(f"Datei existiert: {exists}")

    conn = sqlite3.connect(path)
    try:
        print("\nTabellen:")
        for t in list_tables(conn):
            print(" -", t[0])
        # Check Weapon table
        print("\n== Weapon (Schema + Sample) ==")
        info = pragma_table_info(conn, "Weapon")
        if info is None:
            print("PRAGMA table_info(Weapon) fehlgeschlagen oder Tabelle fehlt.")
        else:
            print("Spalten (PRAGMA):")
            for col in info:
                print(f"  cid={col[0]} name={col[1]} type={col[2]} notnull={col[3]} dflt_value={col[4]} pk={col[5]}")
        cols, rows = show_table_sample(conn, "Weapon")
        if cols is None:
            print("SELECT * FROM Weapon fehlgeschlagen oder Tabelle fehlt.")
            # show exception for debugging
            try:
                cur = conn.cursor()
                cur.execute("SELECT Id,Name,Description,Damage,BoostHP,BoostATK,BoostDEF FROM Weapon")
                print("Unerwartet: spezielle Abfrage erfolgreich.")
            except Exception as e:
                print("Fehler beim direkten SELECT auf Weapon:")
                traceback.print_exc()
        else:
            print("Spaltennamen:", cols)
            if not rows:
                print("Keine Einträge in Weapon.")
            else:
                print("Erste Zeilen:")
                for r in rows:
                    print(" ", r)

        # Also show legacy table if present
        print("\n== Waffen (Schema + Sample) ==")
        info_w = pragma_table_info(conn, "Waffen")
        if info_w is None:
            print("Tabelle 'Waffen' fehlt oder PRAGMA schlug fehl.")
        else:
            print("Spalten (PRAGMA):")
            for col in info_w:
                print(f"  cid={col[0]} name={col[1]} type={col[2]} notnull={col[3]} dflt_value={col[4]} pk={col[5]}")
            cols_w, rows_w = show_table_sample(conn, "Waffen")
            if cols_w and rows_w is not None:
                print("Spaltennamen:", cols_w)
                if not rows_w:
                    print("Keine Einträge in Waffen.")
                else:
                    print("Erste Zeilen:")
                    for r in rows_w:
                        print(" ", r)
    finally:
        conn.close()

    if args.reset:
        if not exists:
            print("\nDB existiert nicht — init_db() wird aufgerufen, um DB und Seeds zu erstellen.")
            import importlib
            db = importlib.import_module("db")
            db.init_db()
            print("init_db() ausgeführt.")
            return
        print("\n-- Reset: Backup wird erstellt, dann Weapon gelöscht und neu angelegt --")
        backup_db(path)
        reset_weapon_table(path)
        conn2 = sqlite3.connect(path)
        try:
            wc2 = table_count(conn2, "Weapon")
            print("Nach Reset - Weapon Anzahl:", wc2)
        finally:
            conn2.close()

if __name__ == "__main__":
    main()