import tkinter as tk
from tkinter import scrolledtext
from PIL import Image, ImageTk
import os
import db

app_instance = None

# Absolute Pfade festlegen, damit die Sprites gefunden werden können
_PROJECT_DIR = os.path.dirname(os.path.abspath(__file__))
_SPRITES_DIR = os.path.join(_PROJECT_DIR, "sprites")

def ui_print(*args, sep=" ", end="\n", color=None):
    """Schreibt Terminal ausgabe in GUI"""
    msg = sep.join(map(str, args)) + end
    if app_instance and getattr(app_instance, "textbox", None):
        app_instance.write_to_console(msg, color=color)
    else:
        print(msg, end="")

def getSprite(entityName: str) -> str:
    """Ordnet Gegner namen den Sprites zu"""
    lowered = entityName.lower()

    sprite_filename = "sample.png"  # default

    match lowered:
        case "eisgeist":
            sprite_filename = "eisgeist.png"
        case "goblin":
            sprite_filename = "goblin.png"
        case "fledermaus":
            sprite_filename = "fledermaus.png"
        case "goblin.png":
            sprite_filename = "goblin.png"
        case "adler":
            sprite_filename = "greifvogel.png"
        case "hexe":
            sprite_filename = "hexe.png"
        case "lavagolem":
            sprite_filename = "lavagolem.png"
        case "lehrer":
            sprite_filename = "lehrer_versuch.png"
        case "nashorn":
            sprite_filename = "nashorn.png"
        case "schlange":
            sprite_filename = "schlange.png"
        case "sumpfkrabbler":
            sprite_filename = "sumpfkrabbler.png"
        case "lehrer":
            sprite_filename = "lehrer_versuch.png"
        case _: 
            sprite_filename = "sample.png"
    
    return os.path.join(_SPRITES_DIR, sprite_filename)

class GameUI(tk.Tk):
    def __init__(self):
        super().__init__()
        global app_instance
        app_instance = self
        self.title("Integratorgame")
        self.configure(bg="white")
        self.geometry("600x600")
        self.create_widgets()

    def write_to_console(self, msg: str, color=None):
        """in das Scrolltextfeld schreiben"""
        if not getattr(self, "textbox", None):
            return
        self.textbox.configure(state="normal")
        if color:
            self.textbox.insert(tk.END, msg, color)
        else:
            self.textbox.insert(tk.END, msg)
        self.textbox.see(tk.END)
        self.textbox.configure(state="disabled")

    def create_widgets(self):


        # Statliste links aus DB
        self.stats_frame = tk.LabelFrame(self, text="Stats", bg="white")
        self.stats_frame.place(x=5, y=10, width=150, height=250)
        # Labels für updates vorbereiten
        self.stats_labels = {}
        stats_map = [
            ("Lvl", "Level"),
            ("HP", "HP"),
            ("Attack", "Attack"),
            ("Defense", "Defense")
        ]
        for key, label in stats_map:
            lbl = tk.Label(self.stats_frame, text=f"{label}: -", anchor="w", bg="white")
            lbl.pack(fill="x")
            self.stats_labels[key] = lbl
        
        # Gegnername und Sprite
        # Label mit Gegnername
        self.enemy_label = tk.Label(self, text="", bg="white", font=(None, 10, "bold"))
        self.enemy_label.place(x=480, y=35)
        
        # Gegner HP und Atk
        self.enemy_hp_label = tk.Label(self, text="HP: -", bg="white")
        self.enemy_hp_label.place(x=480, y=55)
        self.enemy_attack_label = tk.Label(self, text="ATK: -", bg="white")
        self.enemy_attack_label.place(x=480, y=75)

        # Sprite wird bei jedem gegner aktualisiert
        self.sprite_label = tk.Label(self, text="", bg="white")
        self.sprite_label.place(relx=0.5, y=30, anchor="n")
        self.sprite_img = None  # Store PhotoImage reference

        # ScrolledText (readonly)
        text_box_frame = tk.Frame(self, bg="white")
        text_box_frame.place(x=0, y=270, width=600, height=230)
        self.textbox = scrolledtext.ScrolledText(
            text_box_frame,
            font=("Consolas", 10),
            wrap="word",
            bg="white",
            fg="black",
            insertbackground="white",
            state="disabled"
        )
        self.textbox.pack(fill=tk.BOTH, expand=True)
        
        # Farbmarkierungen setzen
        self.textbox.tag_config("blue", foreground="blue")
        self.textbox.tag_config("red", foreground="red")

        # Input Feld
        self.inputframe = tk.Frame(self, bg="white")
        self.inputframe.place(x=0, y=515, width=600, height=35)
        tk.Label(self.inputframe, text="Eingabe:", bg="white").pack(side=tk.LEFT, padx=(6,2))
        self.inputentry = tk.Entry(self.inputframe, font=("Consolas", 11), bg="white", fg="black")
        self.inputentry.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=6, pady=4)
        # Enter auf ok binden
        self.inputentry.bind("<Return>", lambda e: self.on_submit_input())
        self.inputbutton = tk.Button(self.inputframe, text="OK", command=self.on_submit_input)
        self.inputbutton.pack(side=tk.RIGHT, padx=4)

        # Input fokusieren
        try:
            self.inputentry.focus_set()
        except Exception:
            pass

        # Buttons (Exit links, Angriffe rechts)
        bbar = tk.Frame(self, bg="white")
        bbar.pack(side=tk.BOTTOM, fill=tk.X)
        tk.Button(bbar, text="Exit", bg="#7a3b3b", fg="white", command=self.quit).pack(side=tk.LEFT, ipadx=20)
        # Spacer um Buttons aufzuteilen
        spacer = tk.Frame(bbar, bg="white")
        spacer.pack(side=tk.LEFT, expand=True)
        # Primärknopf
        self.primary_btn = tk.Button(bbar, text="Primär", bg="#8046a8", fg="white", command=self.attack_primary)
        self.primary_btn.pack(side=tk.LEFT, padx=2, ipadx=30)
        # Sekundär und Fähigkeitknöpfe
        self.secondary_btn = tk.Button(bbar, text="Sekundär", bg="#8046a8", fg="white", command=self.attack_secondary)
        self.secondary_btn.pack(side=tk.LEFT, padx=2, ipadx=30)
        self.ability_btn = tk.Button(bbar, text="Fähigkeit", bg="#8046a8", fg="white", command=self.attack_ability)
        self.ability_btn.pack(side=tk.LEFT, padx=2, ipadx=30)

    def on_submit_input(self, event=None):
        """Input handler"""
        user_input = self.inputentry.get()
        self.inputentry.delete(0, tk.END)
        # Übergib Input an Spiellogik
        if getattr(self, "main3_instance", None):
            try:
                self.main3_instance.handle_user_input(user_input)
            except Exception:
                ui_print("Fehler beim Verarbeiten der Eingabe")
        else:
            ui_print(user_input)

    def update_stats(self, username: str):
        """Aktualisieren der Stat anzeigen"""
        if not username:
            return
        self._current_player_name = username
        try:
            prow = db.get_player(username)
            if not prow:
                # Fehlerhandling
                for lbl in self.stats_labels.values():
                    lbl.configure(text="-")
                tk.Label(self.stats_frame, text="Spieler nicht gefunden", bg="white").pack(anchor="w")
                return
            # DB Felder auf die Labelfelder mappen
            mapping = {
                "Lvl": "Lvl",
                "HP": "HP",
                "Attack": "Attack",
                "Defense": "Defense",
                "XP": "XP"
            }
            for k, lbl in self.stats_labels.items():
                db_key = mapping.get(k, k)
                val = prow.get(db_key) if isinstance(prow, dict) else None
                # Alle Zahlen als Integer ausgeben
                if isinstance(val, (int, float)):
                    try:
                        val_display = int(val)
                    except Exception:
                        val_display = val
                else:
                    val_display = val
                lbl.configure(text=f"{k}: {val_display}")
        except Exception:
            ui_print("Fehler beim Aktualisieren der Stats\n")

    def refresh_stats(self):
        """Stats für Spieler aktualisieren"""
        if hasattr(self, "_current_player_name"):
            self.update_stats(self._current_player_name)

    def set_player_stats(self, hp: int, attack: int, defense: int, xp: int, lvl: int):
        """Spieler stats mit prozessdaten aktualisieren.
            Dadurch muss die Datenbank nicht immer neu gelesen werden.
        """
        try:
            # Nur die benötigten Label aktualisieren
            if "HP" in self.stats_labels:
                self.stats_labels["HP"].configure(text=f"HP: {int(hp)}")
            if "Attack" in self.stats_labels:
                self.stats_labels["Attack"].configure(text=f"Attack: {int(attack)}")
            if "Defense" in self.stats_labels:
                self.stats_labels["Defense"].configure(text=f"Defense: {int(defense)}")
            if "XP" in self.stats_labels:
                self.stats_labels["XP"].configure(text=f"XP: {int(xp)}")
            if "Lvl" in self.stats_labels:
                self.stats_labels["Lvl"].configure(text=f"Lvl: {int(lvl)}")
        except Exception:
            pass

    def set_enemy_name(self, name: str):
        """Gegner name setzen"""
        try:
            self.enemy_label.configure(text=str(name) if name else "")
            if not name:
                # Stats clearen
                self.enemy_hp_label.configure(text="HP: -")
                self.enemy_attack_label.configure(text="ATK: -")
        except Exception:
            pass

    def set_enemy_stats(self, hp: int, attack: int):
        """Gegner Stats updaten"""
        try:
            self.enemy_hp_label.configure(text=f"HP: {hp}")
            self.enemy_attack_label.configure(text=f"ATK: {attack}")
        except Exception:
            pass

    def update_enemy_sprite(self, enemy_name: str):
        """Sprite updaten"""
        try:
            sprite_path = getSprite(enemy_name)
            img = Image.open(sprite_path).resize((256, 256), Image.NEAREST)
            self.sprite_img = ImageTk.PhotoImage(img)
            self.sprite_label.configure(image=self.sprite_img)
        except Exception:
            self.sprite_label.configure(text="Sprite fehlt!", image="")
            self.sprite_img = None

    def attack_primary(self):
        """Primärattacke an Main3 übergeben"""
        if getattr(self, "main3_instance", None):
            try:
                self.main3_instance.handle_attack("primary")
            except Exception:
                ui_print("Fehler beim Ausführen von attack_primary")
        else:
            ui_print("Primary attack (no game instance)")

    def attack_secondary(self):
        """Sekundärattacke an Main3 übergeben"""
        if getattr(self, "main3_instance", None):
            try:
                self.main3_instance.handle_attack("secondary")
            except Exception:
                ui_print("Fehler beim Ausführen von attack_secondary")
        else:
            ui_print("Secondary attack (no game instance)")

    def attack_ability(self):
        """Fähigkeit an Main3 übergeben"""
        if getattr(self, "main3_instance", None):
            try:
                self.main3_instance.handle_attack("ability")
            except Exception:
                ui_print("Fehler beim Ausführen von attack_ability")
        else:
            ui_print("Ability attack (no game instance)")

if __name__ == "__main__":
    app = GameUI()
    app.mainloop()
