﻿import sqlite3
import os
from typing import Optional, Dict

# Absoluten Pfad festlegen.
_PROJECT_DIR = os.path.dirname(os.path.abspath(__file__))
_DB_PATH = os.path.join(_PROJECT_DIR, "game.db")

# Verbindung zur SQLite‑DB (Datei game.db)
connection = sqlite3.connect(_DB_PATH, check_same_thread=False)
cursor = connection.cursor()

# Weapon Primärschlüssel finden
_WEAPON_PK_COL = None
def _detect_weapon_pk_col():
    global _WEAPON_PK_COL
    if _WEAPON_PK_COL is not None:
        return _WEAPON_PK_COL
    try:
        cur = connection.cursor()
        cur.execute("PRAGMA table_info(Weapon)")
        cols = [r[1] for r in cur.fetchall()]
        if "Id" in cols:
            _WEAPON_PK_COL = "Id"
        elif "Weapon_ID" in cols:
            _WEAPON_PK_COL = "Weapon_ID"
        else:
            _WEAPON_PK_COL = None
    except Exception:
        _WEAPON_PK_COL = None
    return _WEAPON_PK_COL

def init_db():
    """
    Die 3 Tabellen, Spieler, Gegner, Weapon werden angelegt
    und mit Standardwerten gefüttert.
    REAL ist in sqlite das gleiche wie ein Float
    """
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS Spieler (
        UserName TEXT PRIMARY KEY,
        HP INTEGER,
        Attack INTEGER,
        Defense INTEGER,
        XP INTEGER,
        Lvl INTEGER,
        Weapon TEXT,
        Role TEXT
    )
    """)
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS Gegner (
        Name TEXT PRIMARY KEY,
        HP INTEGER,
        Attack INTEGER,
        Defense INTEGER,
        XP INTEGER,
        Lvl INTEGER
    )
    """)
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS Weapon (
        Id INTEGER PRIMARY KEY AUTOINCREMENT,
        Name TEXT UNIQUE,
        Description TEXT,
        Damage REAL,
        BoostHP INTEGER,
        BoostATK REAL,
        BoostDEF REAL
    )
    """)
    # Über einen Unique index, wird sichergestellt, dass keine Doppelten Einträge enstehen
    try:
        # Doppelte Einträge bereinigen (rowid ist der Primärschlüssel der DB)
        cursor.execute("DELETE FROM Weapon WHERE rowid NOT IN (SELECT MIN(rowid) FROM Weapon GROUP BY Name)")
        connection.commit()
    except Exception:
        # Bei Fehler läuft das Programm weiter
        pass
    try:
        # Unique Index wird für den Waffennamen erstellt, dadurch werden doppelte Einträge verhindert
        cursor.execute("CREATE UNIQUE INDEX IF NOT EXISTS idx_weapon_name ON Weapon(Name)")
        connection.commit()
    except Exception:
        # Bei Fehler läuft das Programm weiter
        pass
    # Commit speichert die Änderungen in der Datenbank dauerhaft, also über mehrere Ausführungen des Programms
    connection.commit()
    seed_weapons()
    seed_enemies()

def seed_weapons():
    """
    Lege die Waffen für die Klassen an. 
    Durch INSERT OR IGNORE werden nur nicht vorhandene Einträge eingefügt
    """
    cursor.executemany(
        "INSERT OR IGNORE INTO Weapon(Name,Description,Damage,BoostHP,BoostATK,BoostDEF) VALUES (?,?,?,?,?,?)",
        [
            ("Kurzschwert", "Leichte Klinge, gut ausgewogen", 5, 0, 1, 0),
            ("Zauberstab", "Verstärkt HP und Magie", 8, 10, 0, 0),
            ("Kriegshammer", "Massiver Hammer, sehr hoher Schaden", 15, 0, 3, 0)
        ]
    )
    connection.commit()

def seed_enemies():
    """
    Legt Initial Gegner an.
    """
    cursor.execute("SELECT COUNT(*) FROM Gegner")
    count = cursor.fetchone()[0]
    if count == 0:
        enemies = [
            ("Goblin", 30, 8, 0, 10, 1),
            ("Eisgeist", 40, 10, 0, 15, 1),
            ("Fledermaus", 50, 12, 0, 25, 2),
            ("Adler", 70, 15, 0, 40, 3),
            ("Hexe", 90, 20, 0, 100, 5),
            ("Lavagolem", 150, 8, 0, 150, 7),
            ("Nashorn", 120, 22, 0, 200, 8),
            ("Schlange", 60, 18, 0, 30, 4),
            ("Sumpfkrabbler", 80, 16, 0, 60, 4),
            ("Lehrer", 300, 30, 0, 300, 10)
        ]
        cursor.executemany(
            "INSERT INTO Gegner(Name,HP,Attack,Defense,XP,Lvl) VALUES (?,?,?,?,?,?)",
            enemies
        )
        connection.commit()

def create_weapon(name: str, description: str, damage: int, boost_hp: int = 0, boost_atk: int = 0, boost_def: int = 0) -> Optional[int]:
    """
    Erstellt eine Neue Waffe, wird aber nicht benutzt.
    Rückgabe: Id der Waffe oder None.
    """
    cursor.execute("""
    INSERT OR IGNORE INTO Weapon(Name,Description,Damage,BoostHP,BoostATK,BoostDEF)
    VALUES (?,?,?,?,?,?)
    """, (name, description, damage, boost_hp, boost_atk, boost_def))
    connection.commit()
    # Gibt den Primärschlüssel der Waffe zurück (Name)
    pk = _detect_weapon_pk_col()
    if pk:
        cursor.execute(f"SELECT {pk} FROM Weapon WHERE Name = ?", (name,))
        r = cursor.fetchone()
        return r[0] if r else None
    # Fallback: Stattdessen nach Name suchen
    cursor.execute("SELECT Name FROM Weapon WHERE Name = ?", (name,))
    r = cursor.fetchone()
    return None

def get_weapon_by_id(wid: int) -> Optional[Dict]:
    """
    Liefert ein Dict mit Waffen-Stats zur gegebenen Id oder None.
    """
    pk = _detect_weapon_pk_col()
    if pk:
        cursor.execute(f"SELECT {pk} as Id,Name,Description,Damage,BoostHP,BoostATK,BoostDEF FROM Weapon WHERE {pk} = ?", (wid,))
    else:
        cursor.execute("SELECT rowid as Id,Name,Description,Damage,BoostHP,BoostATK,BoostDEF FROM Weapon WHERE rowid = ?", (wid,))
    row = cursor.fetchone()
    if not row:
        return None
    return {
        "Id": row[0],
        "Name": row[1],
        "Description": row[2],
        "Damage": row[3],
        "BoostHP": row[4],
        "BoostATK": row[5],
        "BoostDEF": row[6]
    }

def get_weapon_by_name(name: str) -> Optional[Dict]:
    """
    Liefert ein Dict mit Waffen-Stats nach Waffenname oder None.
    """
    pk = _detect_weapon_pk_col()
    if pk:
        cursor.execute(f"SELECT {pk} as Id,Name,Description,Damage,BoostHP,BoostATK,BoostDEF FROM Weapon WHERE Name = ?", (name,))
    else:
        cursor.execute("SELECT rowid as Id,Name,Description,Damage,BoostHP,BoostATK,BoostDEF FROM Weapon WHERE Name = ?", (name,))
    row = cursor.fetchone()
    if not row:
        return None
    return {
        "Id": row[0],
        "Name": row[1],
        "Description": row[2],
        "Damage": row[3],
        "BoostHP": row[4],
        "BoostATK": row[5],
        "BoostDEF": row[6]
    }

def create_player(username: str, hp: int = 100, attack: int = 10, defense: int = 5, xp: int = 0, lvl: int = 1, weapon: Optional[str] = None, role: Optional[str] = None):
    """
    Legt einen Spieler an oder ersetzt einen bestehenden Eintrag.
    weapon und role werden als Strings gespeichert (Namen / Enum.name).
    """
    cursor.execute("""
    INSERT OR REPLACE INTO Spieler(UserName,HP,Attack,Defense,XP,Lvl,Weapon,Role)
    VALUES (?,?,?,?,?,?,?,?)
    """, (username, hp, attack, defense, xp, lvl, weapon, role))
    connection.commit()

def get_player(username: str) -> Optional[Dict]:
    """
    Liest einen Spieler aus der DB und gibt ein Dict oder None zurück.
    Felder: UserName, HP, Attack, Defense, XP, Lvl, Weapon, Role
    """
    cursor.execute("SELECT UserName,HP,Attack,Defense,XP,Lvl,Weapon,Role FROM Spieler WHERE UserName = ?", (username,))
    row = cursor.fetchone()
    if not row:
        return None
    return {
        "UserName": row[0],
        "HP": row[1],
        "Attack": row[2],
        "Defense": row[3],
        "XP": row[4],
        "Lvl": row[5],
        "Weapon": row[6],
        "Role": row[7]
    }

def update_player_level(username: str, new_lvl: int):
    """Aktualisiert das Level eines Spielers in der DB."""
    cursor.execute("UPDATE Spieler SET Lvl = ? WHERE UserName = ?", (new_lvl, username))
    connection.commit()

def update_player_stats(username: str, hp: int, attack: int, defense: int, xp: int, lvl: int, weapon: Optional[str] = None, role: Optional[str] = None):
    """
    Aktualisiert die Spielerwerte. Wenn weapon/role angegeben sind, werden diese Felder auch gesetzt.
    """
    if weapon is None and role is None:
        cursor.execute("""
        UPDATE Spieler SET HP = ?, Attack = ?, Defense = ?, XP = ?, Lvl = ? WHERE UserName = ?
        """, (hp, attack, defense, xp, lvl, username))
    else:
        cursor.execute("""
        UPDATE Spieler SET HP = ?, Attack = ?, Defense = ?, XP = ?, Lvl = ?, Weapon = ?, Role = ? WHERE UserName = ?
        """, (hp, attack, defense, xp, lvl, weapon, role, username))
    connection.commit()

def update_player_equipment(username: str, weapon: Optional[str], role: Optional[str]):
    #Setzt das Weapon‑Feld und Role‑Feld für einen Spieler.
    cursor.execute("UPDATE Spieler SET Weapon = ?, Role = ? WHERE UserName = ?", (weapon, role, username))
    connection.commit()

def get_random_enemy_for_level(player_lvl: int) -> Optional[Dict]:
    """
    Wählt einen zufälligen Gegner aus, der maximal SpielerLvl+1 entspricht (wenn möglich).
    Gibt ein Dict mit Gegnerdaten zurück oder None.
    """
    cursor.execute("SELECT Name,HP,Attack,Defense,XP,Lvl FROM Gegner WHERE Lvl <= ? ORDER BY RANDOM() LIMIT 1", (player_lvl + 1,))
    row = cursor.fetchone()
    if not row:
        cursor.execute("SELECT Name,HP,Attack,Defense,XP,Lvl FROM Gegner ORDER BY RANDOM() LIMIT 1")
        row = cursor.fetchone()
        if not row:
            return None
    return {
        "Name": row[0],
        "HP": row[1],
        "Attack": row[2],
        "Defense": row[3],
        "XP": row[4],
        "Lvl": row[5]
    }

def delete_player(username: str) -> int:
    """
    Testfunktion zum Aufräumen.
    Löscht einen Spieler anhand des Benutzernamens.
    Rückgabe: Anzahl gelöschter Zeilen (0 = nicht gefunden, 1 = gelöscht).
    """
    cursor.execute("DELETE FROM Spieler WHERE UserName = ?", (username,))
    connection.commit()
    return cursor.rowcount