﻿#!/usr/bin/env python3
import os
import time
import builtins
import threading
import queue
import gamegui

# Verzögerung in Sekunden nach jedem print-Aufruf, damit Python nicht zu schnell ausgibt
try:
    _DELAY = float(os.environ.get("GAME_PRINT_DELAY", "0.15"))
except Exception:
    _DELAY = 0.05

_original_print = builtins.print

def _delayed_print(*args, **kwargs):

 # Ersatz für builtins.print, fügt nach jedem regulären print() eine kurze Pause hinzu

    _original_print(*args, **kwargs)
    end = kwargs.get("end", "\n")
    if end != "":
        time.sleep(_DELAY)

# Globales print ersetzen, damit GUI und Spiel-Thread synchronisiert sind
builtins.print = _delayed_print

from ClassesCollection.Weapon         import Weapon
from ClassesCollection.Player         import Player
from ClassesCollection.Creature       import Creature
from ClassesCollection.Game           import Game
from ClassesCollection.GameManagement import GameManagement
from EnumCollection.RoleType          import RoleType
import db

def choose_role() -> RoleType:
    print("Wähle eine Rolle:")
    print("1) Ritter")
    print("2) Riese")
    print("3) Magier")
    choice = input("Eingabe (1-3, Standard 1=KNIGHT): ").strip()
    return {
        "1": RoleType.KNIGHT,
        "2": RoleType.GIANT,
        "3": RoleType.WIZARD
    }.get(choice, RoleType.KNIGHT)


# Starterwaffen pro Rolle
ROLE_START_WEAPON = {
    RoleType.KNIGHT: "Kurzschwert",
    RoleType.GIANT:  "Kriegshammer",
    RoleType.WIZARD: "Zauberstab"
}

# Starter-Stats pro Rolle: (HP, Attack, Defense)
ROLE_DEFAULT_STATS = {
    RoleType.KNIGHT: (100, 12, 6),
    RoleType.GIANT:  (140, 15, 5),
    RoleType.WIZARD: (80, 18, 3)
}

def main():
    """
    Programmstart:
     - DB initialisieren
     - Spielername einlesen, Spieler laden oder neu anlegen
     - Kampf‑Schleife starten
     - Spielstände beim Beenden in DB schreiben
    """
    print("Starte Programm...")
    db.init_db()

    username = input('Spielername: ').strip()

    # Spieler laden oder neu anlegen
    player_row = db.get_player(username)
    if player_row is None:
        role_type = choose_role()
        # Stats nach Rolle festlegen
        default_hp, default_attack, default_defense = ROLE_DEFAULT_STATS.get(role_type, (100, 10, 5))
        default_xp = 0
        default_lvl = 1

        # Lade Starterwaffe aus DB
        start_weapon = None
        start_name = ROLE_START_WEAPON.get(role_type)
        if start_name:
            wrow = db.get_weapon_by_name(start_name)
            if wrow:
                start_weapon = Weapon.from_db(wrow)

        # Spieler in DB anlegen 
        db.create_player(username, default_hp, default_attack, default_defense, default_xp, default_lvl,
                         getattr(start_weapon, "name", None), role_type.name)
        player = Player(username, default_hp, default_attack, default_defense, default_xp, default_lvl, start_weapon, role_type)
        print(f"Neuer Spieler '{username}' erstellt (Lvl {default_lvl}) mit Waffe: {getattr(start_weapon, 'name', 'keine')}.")
        # übergebe GUI die neuen Spielerstats, damit sie angezeigt werden können -> weitermachen falls es nicht klappt
        try:
            if 'controller' in globals() and getattr(globals()['controller'], 'gui', None):
                globals()['controller'].gui.update_stats(username)
        except Exception:
            pass
    else:
        # Rolle aus DB laden (sofern gespeichert), ansonsten neu wählen
        role_name = player_row.get("Role") if isinstance(player_row, dict) else None
        if role_name:
            try:
                role_type = RoleType[role_name]
            except Exception:
                role_type = choose_role()
        else:
            role_type = choose_role()

        # Falls eine Waffe gespeichert ist, lade Weapon‑Objekt
        weapon_obj = None
        weapon_name = player_row.get("Weapon") if isinstance(player_row, dict) else None
        if weapon_name:
            wrow = db.get_weapon_by_name(weapon_name)
            if wrow:
                weapon_obj = Weapon.from_db(wrow)

        player = Player(player_row["UserName"], player_row["HP"], player_row["Attack"], player_row["Defense"],
                        player_row["XP"], player_row["Lvl"], weapon_obj, role_type)
        print(f"Willkommen zurück, {username} (Lvl {player_row['Lvl']}). Deine Waffe: {getattr(weapon_obj, 'name', 'keine')}")
        # Werte an GUI übergeben, damit sie angezeigt werden können -> weitermachen falls es nicht klappt
        try:
            if 'controller' in globals() and getattr(globals()['controller'], 'gui', None):
                globals()['controller'].gui.update_stats(player.getName())
        except Exception:
            pass

    gm = GameManagement()

    # Hauptloop: zufälligen Gegner holen und Kampf starten
    while True:
        enemy_row = db.get_random_enemy_for_level(player.getLvl())

        # Gegner-Objekt instanziieren
        enemy = Creature(enemy_row["Name"], enemy_row["HP"], enemy_row["Attack"], enemy_row["Defense"], enemy_row["XP"], enemy_row["Lvl"], None)
        print(f"Ein Gegner erscheint: {enemy.getName()} (Lvl {enemy_row['Lvl']}, HP {enemy_row['HP']})")
        # Gegner-Info an GUI senden, damit sie angezeigt werden können -> weitermachen falls es nicht klappt
        try:
            if 'controller' in globals() and getattr(globals()['controller'], 'gui', None):
                globals()['controller'].gui.set_enemy_name(enemy.getName())
                globals()['controller'].gui.update_enemy_sprite(enemy.getName())
                globals()['controller'].gui.refresh_enemy_stats(enemy.getHP(), enemy.getAttack(), enemy.getLvl())
        except Exception:
            pass

        game = Game(player, enemy, gm)
        gm.startGame(game)
        # Start Kampf
        try:
            game.start()
        except Exception as e: # Noch nicht passiert!
            print(f"Fehler im Spielablauf: {e}")
            break

        winner = game.winner()
        if winner is None:
            print("Kein Gewinner - unklarer Zustand.") # Noch nicht passiert!
        elif winner.getName() == player.getName():
            # Gewinn: Level up und DB aktualisieren
            print(f"Du hast {enemy.getName()} besiegt! Du bekommst 1 Level.")
            new_lvl = player.getLvl() + 1
            player.setLvl(new_lvl)
            db.update_player_level(player.getName(), new_lvl)
            print(f"Neues Level: {new_lvl}")
            # GUI-Stats aktualisieren
            try:
                if 'controller' in globals() and getattr(globals()['controller'], 'gui', None):
                    globals()['controller'].gui.refresh_stats()
            except Exception:
                pass
        else:
            print("Du wurdest besiegt. Spiel beendet.")
            break

        # Weiterkämpfen‑Abfrage
        cont = input("Weiter kämpfen? (j/n) [j]: ").strip().lower()
        if cont == "n":
            # Spielerwerte in DB speichern und Spiel beenden
            db.update_player_stats(
                player.getName(),
                player.getHP(),
                player.getAttack(),
                player.getDefense(),
                player.getXP(),
                player.getLvl(),
                getattr(player.getWeapon(), "name", None),
                player.getRole().getRoleType().name
            )
            print("Spiel wird beendet. Spielerdaten gespeichert. \nBeende mit \"Exit\".")
            break

class Main3Controller:
    """Controller, der GUI und Spiel-Thread verbindet via einer Input-Queue.

    - handle_user_input(text) -> sobald der Benutzer Text in die GUI eingibt
    - handle_attack(kind) -> Knopf in der GUI gedrückt 
    - get_input(prompt) -> ersetzt builtins.input() im Spiel-Thread, blockiert bis Eingabe da ist
    """

    def __init__(self):
        self._inq = queue.Queue()

    def handle_user_input(self, text: str):
        # Füge die Benutzereingabe der Queue hinzu
        try:
            self._inq.put(text)
        except Exception:
            gamegui.ui_print("Fehler beim Senden der Eingabe an die Spiel-Queue\n")

    def handle_attack(self, kind: str):
        # setze die Aktion in die Queue
        map_choice = {
            "primary": "1",
            "secondary": "2",
            "ability": "3"
        }
        val = map_choice.get(kind, "1")
        # Aktion im UI protokollieren
        gamegui.ui_print(f">>> Aktion: {kind} ({val})\n")
        self._inq.put(val)

    def get_input(self, prompt: str = "") -> str:
        # zeige Prompt in der GUI und warte auf Eingabe aus der Queue
        if prompt:
            # stelle Prompt in der GUI dar
            gamegui.ui_print(prompt)
        try:
            val = self._inq.get(block=True)
            return val
        except Exception:
            return ""

if __name__ == "__main__":
    # starte Main3Controller
    controller = Main3Controller()

    # print in gamegui.ui_print umleiten
    try:
        globals()['_original_print'] = gamegui.ui_print
        builtins.print = _delayed_print
    except Exception:
        # falls es nicht klappt, einfach weitermachen :)
        pass

    # builtins.input auf controller.get_input umleiten 
    builtins.input = controller.get_input

    # startet Spiel-Thread
    def game_thread_fn():
        try:
            main()
        except Exception as e:
            gamegui.ui_print(f"Fehler im Spiel-Thread: {e}\n")

    gt = threading.Thread(target=game_thread_fn, daemon=True)

    # erstellt und startet GUI
    app = gamegui.GameUI()
    app.main3_instance = controller
    # gibt Controller die GUI-Referenz
    controller.gui = app

    # nachdem die GUI läuft, starte den Spiel-Thread
    gt.start()

    app.mainloop()