﻿# Weapon Class
from typing import Optional, List

class Weapon:
    """
    Repräsentation einer Waffe:
    - name: Anzeigename
    - description: kurze Beschreibung
    - damage: Zusätzlicher Schadenswert für Funktion attack() von Entity
    - boosts: Array von Statuswerten, die bei der primary() erhöht werden [hp,attack,defense] 
    - id: optionaler Primärschlüssel aus DB (falls geladen)
    """

    def __init__(self, name: str, description: str, damage: int, boosts: List[int], wid: Optional[int] = None):
        self.id = wid
        self.name = name
        self.description = description
        self.damage = int(damage)
        # boosts: [HP, ATK, DEF]
        self.boosts = [int(b) for b in boosts]

    def primary(self) -> int:
        """
        Primärer Angriffswert der Waffe (wird zu Entity.getAttack() addiert).
        """
        return self.damage

    def secondary(self) -> List[int]:
        """
        Sekundäre Effekte der Waffe: Rückgabe der Boost‑Liste [HP, ATK, DEF].
        Diese Werte werden temporär über Stats.add() angewendet.
        Boostet temporär die Statuswerte des Entities
        """
        return self.boosts

    @classmethod
    def from_db(cls, row: dict) -> "Weapon":
        """
        Hilfsmethode: erstellt ein Weaponobjekt für das jeweilige Entityobjekt (In Datenbank hinterlegt)
        Erwartet Keys: Name, Description, Damage, BoostHP, BoostATK, BoostDEF, Id optional.
        """
        if row is None:
            return None
        return cls(
            name=row.get("Name"),
            description=row.get("Description") or "",
            damage=row.get("Damage") or 0,
            boosts=[row.get("BoostHP") or 0, row.get("BoostATK") or 0, row.get("BoostDEF") or 0],
            wid=row.get("Id")
        )

    def __repr__(self):
        """Lesbare Repräsentation für Debugging."""
        return f"Weapon(id={self.id}, name={self.name}, dmg={self.damage}, boosts={self.boosts})"
