﻿from __future__ import annotations
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from .Entity import Entity

class Stats:
    """
    Verwaltung von Basiswerten und temporären Boosts für ein Entity.
    Temporäre Boosts gelten nur für die Dauer eines Kampfes und werden über
    clear_temporary_boosts() wieder entfernt.
    """

    def __init__(self, entity: "Entity", hp: int, attack: float, defense: float):
        """
        Initialisiert das Stats‑Objekt.

        Args:
            entity: Referenz zur übergeordneten Entity (Player/Creature).
            hp: Start‑HP (Basis).
            attack: Basis‑Angriffswert.
            defense: Basis‑Verteidigung.
        """
        self.entity = entity
        # Basiswerte (als Referenz gespeichert)
        self.hp = int(hp)
        self.attack = float(attack)
        self.defense = float(defense)
        # Interner Zustand: kumulierte temporäre Boosts [HP, ATK, DEF]
        self._temp_boosts = [0, 0, 0]

    def add(self, hp: int, attack: float, defense: float) -> None:
        """
        Fügt temporäre Boosts hinzu (nur für die Dauer des laufenden Kampfes).
        - hp: erhöht maxHP und aktuelle HP temporär.
        - attack/defense: erhöhen die entsprechenden Werte temporär.
        """
        # HP-Boost: maxHP und aktuelle HP erhöhen
        if hp:
            if hasattr(self.entity, "_maxHP"):
                self.entity._maxHP += int(hp)
            self.entity.setHP(self.entity.getHP() + int(hp))
            self._temp_boosts[0] += int(hp)

        if attack:
            self.entity.setAttack(self.entity.getAttack() + attack)
            self._temp_boosts[1] += attack

        if defense:
            self.entity.setDefense(self.entity.getDefense() + defense)
            self._temp_boosts[2] += defense

    def reduce(self, hp: int, attack: float, defense: float) -> None:
        """
        Permanente Reduktion von Werten (im Gegensatz zu add(), das temporär ist).
        Wird aktuell nur punktuell verwendet.
        """
        if hp:
            self.entity.setHP(self.entity.getHP() - int(hp))
        if attack:
            self.entity.setAttack(self.entity.getAttack() - attack)
        if defense:
            self.entity.setDefense(self.entity.getDefense() - defense)

    def clear_temporary_boosts(self) -> None:
        """
        Entfernt sämtliche temporären Boosts und stellt die Basiswerte wieder her.
        - Muss am Ende eines Kampfes aufgerufen werden.
        """
        hp_boost, atk_boost, def_boost = self._temp_boosts
        if hp_boost:
            if hasattr(self.entity, "_maxHP"):
                # maxHP reduzieren, mindestens 1
                self.entity._maxHP = max(1, int(self.entity._maxHP - hp_boost))
            # aktuelle HP an neues max anpassen
            if self.entity.getHP() > self.entity.getMaxHP():
                self.entity.setHP(self.entity.getMaxHP())

        if atk_boost:
            self.entity.setAttack(self.entity.getAttack() - atk_boost)
        if def_boost:
            self.entity.setDefense(self.entity.getDefense() - def_boost)

        # Reset der temporären Boosts
        self._temp_boosts = [0, 0, 0]

    def get_temporary_boosts(self):
        """Gibt das Tupel (hp, attack, defense) der aktuell angewendeten temporären Boosts zurück."""
        return tuple(self._temp_boosts)