from __future__ import annotations
from typing import TYPE_CHECKING
import random
from EnumCollection.RoleType import RoleType

if TYPE_CHECKING:
    from .Game import Game
    from .Entity import Entity
    from .Player import Player

# Roleklasse -> Die Rollenklasse enthält Funktionen und Attribute, die sich auf die Spielerrolle beziehen
class Role:
    """
    Role verwaltet die rollen­spezifischen Fähigkeiten und Zustände eines Spielers.
    - Hält Referenz auf das zugehörige Player‑Objekt.
    - Speichert den RoleType (KNIGHT, GIANT, WIZARD).
    - Verwaltet einen einfachen Cooldown‑Zähler (lastUsedAbility).
    """

    def __init__(self, player: "Player", roleType: RoleType):
        """
        Konstruktor

        Args:
            player: das Player‑Objekt, dem diese Rolle zugeordnet ist.
            roleType: Enum (RoleType), bestimmt das Verhalten der Fähigkeit.
        """
        self.player = player
        self.roleType = roleType
        # statischer Zugriff auf die Stats des Spielers (Referenz)
        self.stats = self.player.getStats()
        # Anzahl Runden seit der letzten Nutzung der Fähigkeit (Cooldown)
        self.lastUsedAbility = 0  # Ability kann z.B. alle 3 Runden genutzt werden

    def getPlayer(self):
        """Gibt das Player‑Objekt zurück."""
        return self.player

    def getRoleType(self):
        """Gibt den RoleType (Enum) zurück."""
        return self.roleType

    def getStats(self):
        """Gibt das Stats‑Objekt des Spielers zurück."""
        return self.stats

    def getLastUsedAbility(self):
        """Gibt die Anzahl Runden seit letzter Fähigkeitennutzung zurück."""
        return self.lastUsedAbility

    def setLastUsedAbility(self, round: int):
        """Setzt den internen Counter für letzte Fähigkeitsnutzung."""
        self.lastUsedAbility = round

    def canUseAbility(self, game: "Game"):
        """
        Prüft, ob die Rolle ihre Fähigkeit benutzen darf.
        Standardregel: Fähigkeit ist verfügbar, wenn lastUsedAbility >= 3.
        """
        return self.lastUsedAbility >= 3

    def useAbility(self, game: "Game", damager: "Entity", target: "Entity"):
        """
        Führt die rollen­spezifische Fähigkeit aus.
        Vor Ausführen wird canUseAbility() geprüft; danach wird der Counter zurückgesetzt.

        Wirkungsbeschreibung (aktuell):
         - KNIGHT: moderater Schaden + moderate Selbstheilung (Kombination aus Giant und Wizard,
                   Heileffekt kleiner als Wizard, Schaden kleiner als Giant).
         - GIANT: hoher Schaden an Ziel und Rückstoß‑Schaden am Anwender.
         - WIZARD: heilt den Anwender um einen Teil des MaxHP.
        """
        if not self.canUseAbility(game):
            return

        # Reset Cooldown-Counter
        self.setLastUsedAbility(0)

        match self.getRoleType():
            case RoleType.KNIGHT:
                # Knight: Kombination aus moderatem Schaden und schwächerer Heilung
                # Schaden: zufällig zwischen 15 und 40 (kleiner als Giant)
                damageToEnemy = random.randint(15, 40)
                # Heilung: 20% der MaxHP (kleiner als Wizard's 35%)
                heal_amount = int(damager.getMaxHP() * 0.20)
                # Anwenden: Schaden auf Gegner, Heilung auf sich (clamped an MaxHP)
                target.defend(damageToEnemy, game)
                new_hp = min(damager.getHP() + heal_amount, damager.getMaxHP())
                damager.setHP(new_hp)
                print(f"{damager.getName()} heilt sich um {heal_amount} HP und verursacht {damageToEnemy} Schaden.")
            case RoleType.GIANT:
                # Starker Schaden an Gegner, geringer Selbstschaden
                damageToEnemy = random.randint(30, 80)
                damageToSelf = random.randint(5, 10)
                target.defend(damageToEnemy, game)
                damager.defend(damageToSelf, game)
            case RoleType.WIZARD:
                # Heile einen Prozentsatz der maximalen HP
                heal_amount = int(damager.getMaxHP() * 0.35)
                new_hp = min(damager.getHP() + heal_amount, damager.getMaxHP())
                damager.setHP(new_hp)
            case _:
                # Safety net: ungültiger RoleType
                # Standard, falls Rolle nicht existiert
                print("Ungültig!")
