﻿from __future__ import annotations
import random
from EnumCollection.AttackType import AttackType

class Game:
    """
    Spielverwaltung für ein Duell: hält Player, Creature, Rundenzähler und Spielstatus.
    Verantwortlich für:
     - Start/Schleife des Kampfes
     - Turnwechsel
     - Anzeige des Zustands (HP etc.)
     - Auslösen von XP‑Zuweisung und Bereinigung nach Kampfende
    """

    def __init__(self, player: "Player", enemy: "Creature", gameManagement: "GameManagement"):
        """
        Konstruktor.

        Args:
            player: Spielerobjekt.
            enemy: Gegnerobjekt.
            gameManagement: Manager, der das Spiel verwaltet.
        """
        self.player = player
        self.enemy = enemy
        self.gameManagement = gameManagement
        # Aktueller Akteur (Player oder Enemy)
        self.current_turn = None
        # Flag ob Kampf beendet
        self.has_ended = False
        # Rundenzähler (1,2,3,...)
        self.round = 0

    def getRound(self):
        """Gibt die aktuelle Rundennummer zurück."""
        return self.round

    def getGameManagement(self):
        """Gibt das GameManagement‑Objekt zurück."""
        return self.gameManagement

    def getPlayer(self):
        """Gibt das Player‑Objekt zurück."""
        return self.player

    def getEnemy(self):
        """Gibt das Enemy‑Objekt zurück."""
        return self.enemy

    def start(self):
        """
        Hauptschleife des Kampfes:
         - Bestimmt zufällig den ersten Zug
         - Solange beide Charaktere HP > 0: führe Züge aus
         - Fragt bei Spielerzug nach Aktion (PRIMARY/SECONDARY/ABILITY)
         - Zufällige Wahl durch Gegner
         - Nach jedem Zug Statusausgabe
        """
        # Import gamegui um farblichen Text zu erstellen
        try:
            from gamegui import ui_print
        except Exception:
            # Fallback: benutze eingebauten print
            ui_print = print
        
        self.firstTurn()
        print(f"Spiel startet: {self.getPlayer().getName()} vs {self.getEnemy().getName()}")
        while self.gameCondition():
            self.round += 1
            print(f"\n--- Runde {self.round} ---")

            # Fähigkeitencooldown erhöhen (zählt Runden seit letzter Nutzung)
            self.getPlayer().getRole().setLastUsedAbility(
                self.getPlayer().getRole().getLastUsedAbility() + 1
            )

            # Anzeige wer gerade am Zug ist (mit Farben)
            if self.current_turn == self.getPlayer():
                # Player turn: blue
                ui_print(f"Aktueller Zug: Player ({self.current_turn.getName()})", color="blue")
            else:
                # Enemy turn: red
                ui_print(f"Aktueller Zug: Enemy ({self.current_turn.getName()})", color="red")

            if self.current_turn == self.getPlayer():
                # Spieler wählt Aktion interaktiv
                role = self.getPlayer().getRole()
                last_used = role.getLastUsedAbility()
                can_use_ability = role.canUseAbility(self)
                print(f"Fähigkeit verfügbar: {'Ja' if can_use_ability else 'Nein'} (Runden seit Nutzung: {last_used})")

                while True:
                    choice = input("Wähle Aktion: ").strip()
                    if choice == "":
                        choice = "1"
                    mapping = {"1": AttackType.PRIMARY, "2": AttackType.SECONDARY, "3": AttackType.ABILITY}
                    if choice not in mapping:
                        print("Ungültige Auswahl. Bitte 1, 2 oder 3 eingeben.")
                        continue
                    selected = mapping[choice]
                    if selected == AttackType.ABILITY and not can_use_ability:
                        print("Ability noch nicht verfügbar. Bitte andere Aktion wählen.")
                        continue
                    break

                # Ausführen der gewählten Aktion
                self.getPlayer().attack(self.getEnemy(), selected, self)

            elif self.current_turn == self.getEnemy():
                # KI: zufällige Wahl zwischen PRIMARY und SECONDARY
                choice = random.choice([AttackType.PRIMARY, AttackType.SECONDARY])
                self.getEnemy().attack(self.getPlayer(), choice, self)

            # Statusausgabe nach Aktion
            print(f"Status: {self.getPlayer().getName()} HP={self.getPlayer().getHP()} | {self.getEnemy().getName()} HP={self.getEnemy().getHP()}")
            # Updatet das GUI mit den aktuellen Statuswerten des Gegners
            try:
                import gamegui
                gui_app = getattr(gamegui, 'app_instance', None)
                if gui_app:
                    # update Anzeige der Statuswerte von HP/Attack des Gegners
                    try:
                        gui_app.set_enemy_stats(self.getEnemy().getHP(), self.getEnemy().getAttack())
                    except Exception:
                        pass
                    # Aktualisiere die Statuswerte des Spielers in der GUI von  Werten im Arbeitsspeicher
                    try:
                        gui_app.set_player_stats(
                            self.getPlayer().getHP(),
                            self.getPlayer().getAttack(),
                            self.getPlayer().getDefense(),
                            self.getPlayer().getXP(),
                            self.getPlayer().getLvl()
                        )
                    except Exception:
                        # Aktualisiere die Werte des Arbeitsspeichers aus Datenbank, falls es fehlschlägt
                        try:
                            gui_app.refresh_stats()
                        except Exception:
                            pass
            except Exception:
                pass

        # Kampfende
        self.has_ended = True
        self.end()

    def gameCondition(self) -> bool:
        """Prüft, ob beide Akteure noch HP > 0 haben."""
        return self.getPlayer().getHP() > 0 and self.getEnemy().getHP() > 0

    def firstTurn(self):
        """Entscheidet zufällig, wer den ersten Zug hat."""
        deciding = random.randint(1, 2)
        self.current_turn = self.getPlayer() if deciding == 1 else self.getEnemy()

    def turn(self):
        """Wechselt den aktuellen Akteur (von Player zu Enemy oder umgekehrt)."""
        self.current_turn = self.getPlayer() if self.current_turn == self.getEnemy() else self.getEnemy()

    def addXP(self, amount, entity: "Entity"):
        """Fügt dem gegebenen Entity XP hinzu (persistiert nicht automatisch)."""
        entity.setXP(entity.getXP() + amount)

    def hasEnded(self) -> bool:
        """Gibt zurück, ob der Kampf beendet ist (Flag)."""
        return self.has_ended

    def winner(self) -> "Entity | None":
        """Gibt den Sieger zurück, wenn der Kampf beendet ist, sonst None."""
        if not self.hasEnded():
            return None
        return self.getPlayer() if self.getPlayer().getHP() > 0 else self.getEnemy()

    def end(self):
        """
        Bereinigt temporäre Boosts und handhabt End‑Kampf Effekte:
         - Entfernt temporäre Boosts (Stats.clear_temporary_boosts)
         - Heilt Spieler auf MaxHP bei Sieg
         - Vergibt XP an Gewinner
        """
        try:
            self.getPlayer().getStats().clear_temporary_boosts()
        except Exception:
            pass
        try:
            self.getEnemy().getStats().clear_temporary_boosts()
        except Exception:
            pass

        win = self.winner()
        if win == self.getPlayer():
            # Spieler heilen und XP vergeben
            self.getPlayer().setHP(self.getPlayer().getMaxHP())
            print(f"\nSieg: {self.getPlayer().getName()} hat gewonnen und wurde auf Max-HP ({self.getPlayer().getMaxHP()}) geheilt.")
            self.addXP(self.getEnemy().getXP(), self.getPlayer())
        else:
            print(f"\nNiederlage: {self.getEnemy().getName()} hat gewonnen.")
