from __future__ import annotations
from typing import TYPE_CHECKING

from .Stats import Stats
from EnumCollection.AttackType import AttackType

if TYPE_CHECKING:
    from .Game import Game

# Superclass for Player and Enemy
class Entity:
    """
    Basis­klasse für alle im Kampf verwendeten Einheiten (Spieler und Gegner).
    Stellt Getter/Setter für Basiswerte zur Verfügung und implementiert
    gemeinsame Kampfmethoden (attack, defend, boost, switch).
    """

    def __init__(self, name, hp, attack, defense, xp, lvl, weapon):
        """
        Konstruktor

        Args:
            name: Anzeigename der Einheit.
            hp: Start‑HP (int).
            attack: Basis‑Angriffswert.
            defense: Basis‑Verteidigung.
            xp: Erfahrungs­punkte.
            lvl: Level.
            weapon: Weapon‑Objekt oder None.
        """
        self.name = name
        self._hp = hp
        self._attack = attack
        self._defense = defense
        self._xp = xp
        self._lvl = lvl
        self._weapon = weapon
        self._maxHP = hp
        # Stats-Objekt verwalten (temporäre Boosts werden dort gehalten)
        self.stats = Stats(self, self._hp, self._attack, self._defense)
        
    def getName(self):
        """Gibt den Namen der Einheit zurück."""
        return self.name
    
    def getHP(self):
        """Gibt die aktuelle HP der Einheit zurück."""
        return self._hp
    
    def getMaxHP(self):
        """Gibt die aktuelle MaxHP zurück (kann temporär durch Boosts verändert sein)."""
        return self._maxHP
    
    def getAttack(self):
        """Gibt den Angriffswert zurück."""
        return self._attack
    
    def getDefense(self):
        """Gibt den Verteidigungswert zurück."""
        return self._defense
    
    def getXP(self):
        """Gibt die XP zurück."""
        return self._xp
    
    def getLvl(self):
        """Gibt das Level zurück."""
        return self._lvl
    
    def getWeapon(self):
        """
        Gibt das Weapon‑Objekt zurück. Falls keine Waffe gesetzt ist, wird
        eine Standard‑'Fist' Weapon instanziiert und als Singleton zurückgegeben.
        Lokal importiert, um zirkuläre Importe zu vermeiden.
        """
        if getattr(self, "_weapon", None) is None:
            from .Weapon import Weapon
            if not hasattr(Entity, "_default_weapon"):
                Entity._default_weapon = Weapon("Fist", "Default unarmed attack", 0, [0, 0, 0])
            return Entity._default_weapon
        return self._weapon

    def setWeapon(self, weapon):
        """Setzt die Waffe (Weapon‑Objekt)."""
        self._weapon = weapon
    
    def getStats(self):
        """Gibt das zugehörige Stats‑Objekt zurück."""
        return self.stats
    
    def setHP(self, amount):
        """Setzt die aktuelle HP (wird auf >=0 gekappt)."""
        self._hp = max(0, int(amount))
    
    def setAttack(self, amount):
        """Setzt den Angriffswert."""
        self._attack = amount
        
    def setDefense(self, amount):
        """Setzt den Verteidigungswert."""
        self._defense = amount
    
    def setXP(self, amount):
        """Setzt die XP."""
        self._xp = amount
    
    def setLvl(self, amount):
        """Setzt das Level."""
        self._lvl = amount
    
    def lvlUp(self):
        """Erhöht das Level um 1."""
        self.setLvl(self.getLvl() + 1)
    
    def switch(self, game: "Game"):
        """Ruft game.turn() auf, um den Zug zu wechseln."""
        game.turn()

    def boost(self, hpBoost, attackBoost, defenseBoost):
        """Wendet temporäre Boosts über Stats.add() an."""
        self.getStats().add(hpBoost, attackBoost, defenseBoost)
        
    def defend(self, damage, game: "Game"):
        """
        Verteidigungsmechanik:
         - Berechnet effektiven Schaden: max(0, damage - defense)
         - Verringert aktuelle HP entsprechend
         - Gibt eine Konsolenmeldung mit altem und neuem HP‑Wert aus
        """
        try:
            dmg_val = float(damage)
        except Exception:
            dmg_val = 0.0
        effective = max(0, int(dmg_val - self.getDefense()))
        old = self.getHP()
        self.setHP(self.getHP() - effective)
        print(f"{self.getName()} erleidet {effective} Schaden (HP: {old} -> {self.getHP()})")
        
    def attack(self, target: "Entity", attackType: AttackType, game: "Game"):
        """
        Führt einen Angriff gegen target aus.
        - PRIMARY: Direkter Schaden basierend auf Waffe und Attack‑Stat.
        - SECONDARY: Teilweiser Schaden + ggf. Waffenspezifische Boosts.
        - ABILITY: delegiert an Role.useAbility().
        Nach Ausführung wird der Zug gewechselt.
        """
        attacker = self.getName()
        defender = target.getName()
        match attackType:
            case AttackType.PRIMARY:
                dmg = self.getWeapon().primary() + self.getAttack()
                print(f"{attacker} führt PRIMARY-Angriff auf {defender} aus (Basis-Schaden: {dmg}).")
                target.defend(dmg, game)
            case AttackType.SECONDARY:
                boosts = self.getWeapon().secondary()
                try:
                    weapon_primary = int(self.getWeapon().primary())
                except Exception:
                    weapon_primary = 0
                sec_damage = int((weapon_primary * 0.5) + (self.getAttack() * 0.5))
                if sec_damage > 0:
                    print(f"{attacker} führt SECONDARY-Angriff auf {defender} aus (Schaden: {sec_damage}).")
                    target.defend(sec_damage, game)
                if boosts and any(b != 0 for b in boosts):
                    print(f"{attacker} erhält Boosts HP {boosts[0]}, ATK {boosts[1]}, DEF {boosts[2]}.")
                    self.boost(boosts[0], boosts[1], boosts[2])
            case AttackType.ABILITY:
                print(f"{attacker} verwendet ABILITY gegen {defender}.")
                self.getRole().useAbility(game, self, target)
            case _:
                print("Ungültig!")
        # Zugwechsel nach Aktion
        self.switch(game)